<?php
/**
 * @package     Joomla.Site
 * @subpackage  mod_currencypal
 *
 * @copyright   Copyright (C) 2025 CurrencyPal. All rights reserved.
 * @license     GNU General Public License version 2 or later
 *
 * Compatible with Joomla 2.5, 3.x, 4.x, 5.x, and 6.x
 */

defined('_JEXEC') or die;

class ModCurrencyPalHelper
{
    /**
     * Translate a language string (compatible with Joomla 2.5+)
     *
     * @param   string  $key  The language key
     * @return  string
     */
    public static function translate($key)
    {
        // Joomla 4+ uses namespaced Text class
        if (class_exists('Joomla\CMS\Language\Text')) {
            return \Joomla\CMS\Language\Text::_($key);
        }

        // Joomla 2.5/3.x uses JText
        if (class_exists('JText')) {
            return \JText::_($key);
        }

        // Fallback
        return $key;
    }

    /**
     * Get the base URL from params
     *
     * @param   object  $params  Module parameters
     * @return  string
     */
    public static function getBaseUrl($params)
    {
        $baseUrl = $params->get('base_url', 'https://currencypal.net');
        return rtrim($baseUrl, '/');
    }

    /**
     * Get the embed URL for the widget
     *
     * @param   object  $params  Module parameters
     * @return  string
     */
    public static function getEmbedUrl($params)
    {
        $baseUrl = self::getBaseUrl($params);
        $widgetId = $params->get('widget_id', '');

        if (empty($widgetId)) {
            return '';
        }

        return $baseUrl . '/widget/' . htmlspecialchars($widgetId, ENT_QUOTES, 'UTF-8');
    }

    /**
     * Sanitize a dimension value
     *
     * @param   string  $value    The dimension value
     * @param   string  $default  Default value if invalid
     * @return  string
     */
    public static function sanitizeDimension($value, $default)
    {
        $value = trim($value);

        if (empty($value)) {
            return $default;
        }

        if (preg_match('/^(\d+(?:\.\d+)?)(px|%|em|rem|vh|vw)$/i', $value)) {
            return $value;
        }

        if (is_numeric($value)) {
            return $value . 'px';
        }

        if ($value === 'auto') {
            return 'auto';
        }

        return $default;
    }

    /**
     * Get container CSS style
     *
     * @param   string  $maxWidth  Maximum width
     * @return  string
     */
    public static function getContainerStyle($maxWidth)
    {
        return sprintf('max-width: %s; width: 100%%; margin: 0 auto;', htmlspecialchars($maxWidth, ENT_QUOTES, 'UTF-8'));
    }

    /**
     * Get iframe CSS style
     *
     * @param   string  $maxHeight  Maximum height
     * @return  string
     */
    public static function getIframeStyle($maxHeight)
    {
        $style = 'width: 100%; border: none; min-height: 200px;';

        if ($maxHeight !== 'auto') {
            $style .= sprintf(' max-height: %s;', htmlspecialchars($maxHeight, ENT_QUOTES, 'UTF-8'));
        }

        return $style;
    }
}
