<?php
/**
 * Plugin Name: CurrencyPal Widget
 * Plugin URI: https://currencypal.net/downloads
 * Description: Embed CurrencyPal currency exchange rate widgets on your WordPress site using shortcodes or Gutenberg blocks.
 * Version: 1.0.2
 * Author: CurrencyPal
 * Author URI: https://currencypal.net
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: currencypal-widget
 */

if (!defined('ABSPATH')) {
    exit;
}

define('CURRENCYPAL_WIDGET_VERSION', '1.0.2');
define('CURRENCYPAL_WIDGET_PATH', plugin_dir_path(__FILE__));
define('CURRENCYPAL_WIDGET_URL', plugin_dir_url(__FILE__));

class CurrencyPal_Widget {

    private static $instance = null;

    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {
        add_action('admin_menu', [$this, 'add_admin_menu']);
        add_action('admin_init', [$this, 'register_settings']);
        add_shortcode('currencypal', [$this, 'render_shortcode']);
        add_action('init', [$this, 'register_block']);
        add_action('wp_enqueue_scripts', [$this, 'enqueue_scripts_and_styles']);
    }

    public static function get_base_url() {
        $base_url = get_option('currencypal_base_url', 'https://currencypal.net');
        return rtrim($base_url, '/');
    }

    public function add_admin_menu() {
        add_options_page(
            __('CurrencyPal Widget Settings', 'currencypal-widget'),
            __('CurrencyPal Widget', 'currencypal-widget'),
            'manage_options',
            'currencypal-widget',
            [$this, 'render_settings_page']
        );
    }

    public function register_settings() {
        register_setting('currencypal_widget_settings', 'currencypal_base_url', [
            'type' => 'string',
            'sanitize_callback' => 'esc_url_raw',
            'default' => 'https://currencypal.net',
        ]);

        add_settings_section(
            'currencypal_widget_main',
            __('Main Settings', 'currencypal-widget'),
            null,
            'currencypal-widget'
        );

        add_settings_field(
            'currencypal_base_url',
            __('CurrencyPal Base URL', 'currencypal-widget'),
            [$this, 'render_base_url_field'],
            'currencypal-widget',
            'currencypal_widget_main'
        );
    }

    public function render_base_url_field() {
        $value = get_option('currencypal_base_url', 'https://currencypal.net');
        ?>
        <input type="url"
               name="currencypal_base_url"
               value="<?php echo esc_attr($value); ?>"
               class="regular-text"
               placeholder="https://currencypal.net" />
        <p class="description">
            <?php esc_html_e('Enter the base URL of your CurrencyPal installation (without trailing slash).', 'currencypal-widget'); ?>
        </p>
        <?php
    }

    public function render_settings_page() {
        if (!current_user_can('manage_options')) {
            return;
        }
        ?>
        <div class="wrap">
            <h1><?php echo esc_html(get_admin_page_title()); ?></h1>

            <form action="options.php" method="post">
                <?php
                settings_fields('currencypal_widget_settings');
                do_settings_sections('currencypal-widget');
                submit_button(__('Save Settings', 'currencypal-widget'));
                ?>
            </form>

            <hr />

            <h2><?php esc_html_e('Usage Instructions', 'currencypal-widget'); ?></h2>

            <h3><?php esc_html_e('Shortcode', 'currencypal-widget'); ?></h3>
            <p><?php esc_html_e('Use the following shortcode to embed a widget:', 'currencypal-widget'); ?></p>
            <code>[currencypal id="YOUR_WIDGET_ID"]</code>

            <h4><?php esc_html_e('Shortcode Attributes:', 'currencypal-widget'); ?></h4>
            <table class="widefat" style="max-width: 600px;">
                <thead>
                    <tr>
                        <th><?php esc_html_e('Attribute', 'currencypal-widget'); ?></th>
                        <th><?php esc_html_e('Required', 'currencypal-widget'); ?></th>
                        <th><?php esc_html_e('Default', 'currencypal-widget'); ?></th>
                        <th><?php esc_html_e('Description', 'currencypal-widget'); ?></th>
                    </tr>
                </thead>
                <tbody>
                    <tr>
                        <td><code>id</code></td>
                        <td><?php esc_html_e('Yes', 'currencypal-widget'); ?></td>
                        <td>-</td>
                        <td><?php esc_html_e('Your CurrencyPal Widget ID', 'currencypal-widget'); ?></td>
                    </tr>
                    <tr>
                        <td><code>max_width</code></td>
                        <td><?php esc_html_e('No', 'currencypal-widget'); ?></td>
                        <td>100%</td>
                        <td><?php esc_html_e('Maximum width (e.g., 500px, 100%)', 'currencypal-widget'); ?></td>
                    </tr>
                    <tr>
                        <td><code>max_height</code></td>
                        <td><?php esc_html_e('No', 'currencypal-widget'); ?></td>
                        <td>auto</td>
                        <td><?php esc_html_e('Maximum height (e.g., 400px, 50vh)', 'currencypal-widget'); ?></td>
                    </tr>
                </tbody>
            </table>

            <h4><?php esc_html_e('Examples:', 'currencypal-widget'); ?></h4>
            <p><code>[currencypal id="abc123"]</code></p>
            <p><code>[currencypal id="abc123" max_width="500px" max_height="400px"]</code></p>

            <h3><?php esc_html_e('Gutenberg Block', 'currencypal-widget'); ?></h3>
            <p><?php esc_html_e('Search for "CurrencyPal Widget" in the block inserter to add the widget block.', 'currencypal-widget'); ?></p>
        </div>
        <?php
    }

    public function render_shortcode($atts) {
        $atts = shortcode_atts([
            'id' => '',
            'max_width' => '100%',
            'max_height' => 'auto',
        ], $atts, 'currencypal');

        if (empty($atts['id'])) {
            if (current_user_can('edit_posts')) {
                return '<p style="color: red;">' . esc_html__('CurrencyPal Widget Error: Widget ID is required.', 'currencypal-widget') . '</p>';
            }
            return '';
        }

        return self::render_widget($atts['id'], $atts['max_width'], $atts['max_height']);
    }

    public static function render_widget($widget_id, $max_width = '100%', $max_height = 'auto') {
        $widget_id = sanitize_text_field($widget_id);
        $max_width = self::sanitize_dimension($max_width, '100%');
        $max_height = self::sanitize_dimension($max_height, 'auto');

        $base_url = self::get_base_url();
        $embed_url = $base_url . '/widget/' . $widget_id;

        $container_style = sprintf(
            'max-width: %s; width: 100%%;',
            esc_attr($max_width)
        );

        $iframe_style = 'width: 100%; border: none;';
        if ($max_height !== 'auto') {
            $iframe_style .= sprintf(' max-height: %s;', esc_attr($max_height));
        }

        $iframe_id = 'currencypal-iframe-' . esc_attr($widget_id);

        // Enqueue the resize handler script with widget-specific data
        self::enqueue_resize_script($widget_id, $iframe_id);

        ob_start();
        ?>
        <div class="currencypal-widget-container" style="<?php echo esc_attr($container_style); ?>">
            <iframe
                id="<?php echo esc_attr($iframe_id); ?>"
                src="<?php echo esc_url($embed_url); ?>"
                style="<?php echo esc_attr($iframe_style); ?>"
                loading="lazy"
                title="<?php esc_attr_e('CurrencyPal Exchange Rate Widget', 'currencypal-widget'); ?>">
            </iframe>
        </div>
        <?php
        return ob_get_clean();
    }

    private static function enqueue_resize_script($widget_id, $iframe_id) {
        static $script_registered = false;
        static $widgets = [];

        // Register the base script once
        if (!$script_registered) {
            wp_register_script(
                'currencypal-widget-resize',
                false,
                [],
                CURRENCYPAL_WIDGET_VERSION,
                true
            );
            wp_enqueue_script('currencypal-widget-resize');

            // Add the main event listener script
            $main_script = "
                (function() {
                    if (window.currencypalResizeListenerAdded) return;
                    window.currencypalResizeListenerAdded = true;
                    window.addEventListener('message', function(event) {
                        if (event.data && event.data.type === 'currencypal-resize' && event.data.widgetId) {
                            var iframe = document.getElementById('currencypal-iframe-' + event.data.widgetId);
                            if (iframe && event.data.height) {
                                iframe.style.height = event.data.height + 'px';
                            }
                        }
                    });
                })();
            ";
            wp_add_inline_script('currencypal-widget-resize', $main_script);
            $script_registered = true;
        }
    }

    private static function sanitize_dimension($value, $default) {
        $value = trim($value);
        if (empty($value)) {
            return $default;
        }
        if (preg_match('/^(\d+(?:\.\d+)?)(px|%|em|rem|vh|vw)$/i', $value)) {
            return $value;
        }
        if (is_numeric($value)) {
            return $value . 'px';
        }
        if ($value === 'auto') {
            return 'auto';
        }
        return $default;
    }

    public function enqueue_scripts_and_styles() {
        // Register and enqueue a minimal style handle for our inline styles
        wp_register_style(
            'currencypal-widget-styles',
            false,
            [],
            CURRENCYPAL_WIDGET_VERSION
        );
        wp_enqueue_style('currencypal-widget-styles');
        wp_add_inline_style('currencypal-widget-styles', '
            .currencypal-widget-container {
                margin: 0 auto;
            }
            .currencypal-widget-container iframe {
                display: block;
                min-height: 200px;
            }
        ');
    }

    public function register_block() {
        if (!function_exists('register_block_type')) {
            return;
        }

        wp_register_script(
            'currencypal-widget-block',
            CURRENCYPAL_WIDGET_URL . 'block.js',
            ['wp-blocks', 'wp-element', 'wp-editor', 'wp-components', 'wp-i18n'],
            CURRENCYPAL_WIDGET_VERSION,
            true
        );

        wp_localize_script('currencypal-widget-block', 'currencypalWidget', [
            'baseUrl' => self::get_base_url(),
        ]);

        register_block_type('currencypal/widget', [
            'editor_script' => 'currencypal-widget-block',
            'render_callback' => [$this, 'render_block'],
            'attributes' => [
                'widgetId' => [
                    'type' => 'string',
                    'default' => '',
                ],
                'maxWidth' => [
                    'type' => 'string',
                    'default' => '100%',
                ],
                'maxHeight' => [
                    'type' => 'string',
                    'default' => 'auto',
                ],
            ],
        ]);
    }

    public function render_block($attributes) {
        $widget_id = isset($attributes['widgetId']) ? $attributes['widgetId'] : '';
        $max_width = isset($attributes['maxWidth']) ? $attributes['maxWidth'] : '100%';
        $max_height = isset($attributes['maxHeight']) ? $attributes['maxHeight'] : 'auto';

        if (empty($widget_id)) {
            return '';
        }

        return self::render_widget($widget_id, $max_width, $max_height);
    }
}

CurrencyPal_Widget::get_instance();
